<?php

/**
 * This is the model class for table "Szkolenie".
 *
 * The followings are the available columns in table 'Szkolenie':
 * @property double $id
 * @property string $nazwa
 * @property double $idDziedzina
 * @property double $poziom
 * @property string $czyEgzamin
 * @property double $czasTrwania
 * @property double $cena
 * @property string $tresc
 *
 * Relations
 * @propery Dziedzina dziedzina
 * @propery SzkolenieFirmy[] szkolenieFirmies
 * @propery SzkoleniePracownika[] szkoleniePracownikas
 * @propery SzkoleniePytaniaEgzamin[] pytania
 *
 * @property boolean $moodle - goes to moodle or this system
 */
class Szkolenie extends CActiveRecord
{
	public $czySukces = null;

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return Szkolenie the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	public function getMetaData()
	{
		$data = parent::getMetaData();
		$data->columns['czySukces'] = array('name' => 'czySukces');
		return $data;
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'Szkolenie';
	}

	protected function afterFind()
	{
		$this->tresc = OracleHelper::getClobValue('"Szkolenie"', '"tresc"', '"id"', $this->id);
		return parent::afterFind();
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		return array(
			array('idDziedzina', 'required'),
			array('idDziedzina, poziom, czasTrwania, cena', 'numerical'),
			array('nazwa', 'length', 'max'=>64),
			array('czyEgzamin', 'length', 'max'=>1),
			array('tresc', 'length', 'max'=>40000),
			array('idDziedzina, poziom, tresc, nazwa', 'required'),

			array('id, nazwa, idDziedzina, poziom, czyEgzamin, czasTrwania, cena, tresc', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		return array(
			'dziedzina'=>array(self::BELONGS_TO, 'Dziedzina', 'idDziedzina'),
			'szkolenieFirmies'=>array(self::HAS_MANY, 'SzkolenieFirmy', 'idSzkolenia'),
			'szkoleniePracownikas'=>array(self::HAS_MANY, 'SzkoleniePracownika', 'idSzkolenia'),
			'pytania'=>array(self::HAS_MANY, 'SzkoleniePytaniaEgzamin', 'idSzkolenia'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id'=>Yii::t('Szkolenie', 'ID'),
			'nazwa'=>Yii::t('Szkolenie', 'Nazwa'),
			'idDziedzina'=>Yii::t('Szkolenie', 'Dziedzina'),
			'poziom'=>Yii::t('Szkolenie', 'Poziom'),
			'czyEgzamin'=>Yii::t('Szkolenie', 'Egzamin'),
			'czasTrwania'=>Yii::t('Szkolenie', 'Czas trwania [gg:mm:ss]'),
			'cena'=>Yii::t('Szkolenie', 'Cena [zł]'),
			'tresc'=>Yii::t('Szkolenie', 'Treść'),
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		$criteria = new CDbCriteria;

		$criteria->compare(Yii::app()->db->quoteColumnName("t.id"), $this->id);
		$criteria->compare('LOWER("t"."nazwa")', strtolower($this->nazwa), true);
		$criteria->compare('LOWER("dziedzina"."nazwa")', strtolower($this->idDziedzina), true);
		$criteria->compare('"poziom"', $this->getPoziomNum());
		$criteria->compare('"czyEgzamin"', $this->getCzyEgzaminStr());
		$criteria->compare('"czasTrwania"', $this->getCzasTrwaniaNum());
		$criteria->compare('"cena"', $this->cena);
		$criteria->compare('"tresc"', $this->tresc, true);
		// $criteria->compare('LOWER("tresc")', strtolower($this->tresc), true);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	public function getCzyEgzaminStr()
	{
		if (empty($this->czyEgzamin))
			return $this->czyEgzamin;
		if ($this->czyEgzamin == 'T' || $this->czyEgzamin == 't' || $this->czyEgzamin == '1')
			return 1;
		if ($this->czyEgzamin == 'N' || $this->czyEgzamin == 'n' || $this->czyEgzamin == '0')
			return 1;
		return '';
	}

	public function getCzasTrwaniaNum()
	{
		if (empty($this->czasTrwania))
			return '';
		if (is_numeric($this->czasTrwania))
			return $this->czasTrwania;
		$arr = explode(':', $this->czasTrwania);
		$sum = 0;
		for ($i = count($arr)-1, $mul = 1; $i >= 0; $i--, $mul *= 60)
			$sum += (int)$arr[$i] * $mul;
		return $sum;
	}

	public static function getCzasTrwaniaString($czas)
	{
		$result = '';
		$czas = (int)$czas;

		$hh = floor($czas / 3600) % 24;
		if ($hh) $result .= $hh . ' godz. ';

		$mm = floor($czas / 60) % 60;
		if ($mm) $result .= $mm . ' min. ';

		$ss = $czas % 60;
		if ($ss) $result .= $ss . ' sek. ';

		return $result;
	}

	public function getCzasTrwaniaStr()
	{
		return self::getCzasTrwaniaString($this->czasTrwania);
	}

	static private $_poziomy = array(
		1 => 'Łatwy',
		2 => 'Trudny',
		3 => 'Bardzo trudny',
	);

	public static function getPoziomy()
	{
		return self::$_poziomy;
	}

	public function getPoziomStr()
	{
		return self::getPoziomString($this->poziom);
	}

	public static function getPoziomString($poziom)
	{
		return isset(self::$_poziomy[$poziom]) ? self::$_poziomy[$poziom] : '';
	}

	public function getPoziomNum()
	{
		if (empty($this->poziom))
			return '';
		if (is_numeric($this->poziom))
			return $this->poziom;
		$poziom = strtolower($this->poziom);
		foreach (self::$_poziomy as $k => $v)
			if (strpos(strtolower($v), $poziom) !== false)
				return $k;
		return -1;
	}

	public function getEgzamin()
	{
		return $this->czyEgzamin ? 'Tak' : 'Nie';
	}
}
